<?php
/*
Plugin Name: Gitweb
Plugin URI: http://octo.it/yourls-gitweb/
Description: Automatically redirect to a Gitweb installation if an appropriate Git object exists.
Version: 1.0
Author: Florian "octo" Forster
Author URI: http://octo.it/
*/

/**
 * yourls -- Gitweb plugin
 * Copyright (C) 2011  Florian Forster
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   Florian Forster <ff at octo.it>
 **/

function gitweb_check_repository ($obj, $repo, $dir, $base_url) /* {{{ */
{
  $output = array ();
  $retval = 0;

  $cmd = 'git --git-dir=' . escapeshellarg ($dir)
    . ' rev-parse --verify ' . escapeshellarg ($obj)
    . ' 2>/dev/null';
  $obj_name = trim (shell_exec ($cmd));
  if (!$obj_name)
    return (false);

  if (!preg_match ('/^[0-9a-fA-F]{40}$/', $obj_name))
  {
    error_log ("git-rev-parse(1) returned unexpected object name: $obj_name");
    return (false);
  }

  $cmd = 'git --git-dir=' . escapeshellarg ($dir)
    . ' cat-file -t ' . escapeshellarg ($obj_name)
    . ' 2>/dev/null';
  $obj_type = trim (shell_exec ($cmd));
  if (!$obj_type)
  {
    error_log ("gitweb_check_repository: git-cat-file(1) failed.");
    return (false);
  }

  if ($obj_type == 'commit')
  {
    $to_url = "$base_url?p=" . urlencode ($repo) . ';a=commitdiff;h=' . urlencode ($obj_name);
    yourls_redirect ($to_url, /* status = */ 301);
    return (true);
  }
  elseif ($obj_type == 'tag')
  {
    $to_url = "$base_url?p=" . urlencode ($repo) . ';a=tag;h=' . urlencode ($obj_name);
    yourls_redirect ($to_url, /* status = */ 301);
    return (true);

  }
  elseif ($obj_type == 'tree')
  {
    $to_url = "$base_url?p=" . urlencode ($repo) . ";a=tree;h=" . urlencode ($obj_name);
    yourls_redirect ($to_url, /* status = */ 301);
    return (true);
  }
  elseif ($obj_type == 'blob')
  {
    $to_url = "$base_url?p=" . urlencode ($repo) . ";a=blob;h=" . urlencode ($obj_name);
    yourls_redirect ($to_url, /* status = */ 301);
    return (true);
  }
  else
  {
    error_log ("Gitweb plugin: Object \"$obj_name\" in repository \"$repo\" has unknown type \"$obj_type\".");
    return (false);
  }
} /* }}} function gitweb_check_repository */

/* This callback function is called when the given keyword was not found in the 
  * database. I'll see if this looks like an object identifier in a Git 
  * repository and, if so, try to locate the object using the local 
  * repositories. */
function gitweb_redirect_keyword_not_found ($args) /* {{{ */
{
  $keyword = $args[0];

  if (!preg_match ('/^[0-9a-fA-F]{6,40}$/', $keyword))
    return;

  $base_directory = yourls_get_option ('gitweb_base_directory');
  if (!$base_directory)
    return;

  $base_url = yourls_get_option ('gitweb_base_url');
  if (!$base_url)
    return;

  $dh = opendir ($base_directory);
  if (!$dh)
    return;

  while (($subdir = readdir ($dh)) !== false)
  {
    /* Ignore all files and directories starting with a dot, including the 
     * special directories "." and "..". */
    if (substr ($subdir, 0, 1) == '.')
      continue;

    $absdir = "$base_directory/$subdir";
    if (!is_dir ($absdir))
      continue;

    /* Ignore repositories which are private (i.e. not exported by the 
     * git-daemon(1). We might leak information if we don't. */
    if (!file_exists ("$absdir/git-daemon-export-ok"))
      continue;

    if (gitweb_check_repository ($keyword, $subdir, $absdir, $base_url))
      break;
  }

  closedir ($dh);
} /* }}} function gitweb_redirect_keyword_not_found */

function gitweb_set_base_directory ($dir) /* {{{ */
{
  /* Remove trailing slashes. */
  $dir = preg_replace ('/\/+$/', '', $dir);

  if (!preg_match ('/^\//', $dir))
  {
    print ("<p class=\"error\">Not an absolute path: "
      . htmlspecialchars ($dir)
      . "</p>\n");
    return (false);
  }

  if (!is_dir ($dir))
  {
    print ("<p class=\"error\">Not a directory: "
      . htmlspecialchars ($dir)
      . "</p>\n");
    return (false);
  }

  /* Open the directory only to check its permissions. */
  $dh = opendir ($dir);
  if (!$dh)
  {
    print ("<p class=\"error\">Unable to open directory.</p>\n");
    return (false);
  }
  closedir ($dh);

  yourls_update_option ('gitweb_base_directory', $dir);
  return (true);
} /* }}} function gitweb_set_base_directory */

function gitweb_set_base_url ($url) /* {{{ */
{
  if (!preg_match ('/https?:\/\//i', $url))
  {
    print ("<p class=\"error\">This does not look like a valid URL: "
      . htmlspecialchars ($url)
      . "</p>\n");
    return (false);
  }

  $url = preg_replace ('/\?.*/', '', $url);

  yourls_update_option ('gitweb_base_url', $url);
  return (true);
} /* }}} function gitweb_set_base_directory */

function gitweb_show_plugin_page () /* {{{ */
{
  echo <<<HTML
    <h2>Gitweb Plugin Administration Page</h2>
    <p>This plugin redirects to a Gitweb installation if a keyword wasn't
      found in the database, looks like a Git object ID and is found in a
      local Git repository.</p>
HTML;

  if (isset ($_POST['base_directory']))
    gitweb_set_base_directory ($_POST['base_directory']);

  if (isset ($_POST['base_url']))
    gitweb_set_base_url ($_POST['base_url']);

  $base_directory = yourls_get_option ('gitweb_base_directory');
  if ($base_directory)
    $base_directory = htmlspecialchars ($base_directory);

  $base_url = yourls_get_option ('gitweb_base_url');
  if ($base_url)
    $base_url = htmlspecialchars ($base_url);

  echo <<<HTML
    <form method="post">
      <table style="background-color: #cdcdcd; border-spacing: 1px;">
        <tr>
          <th style="border: 1px solid white; background: #C7E7FF; padding: 4px;"><label for="base_directory">Base directory</label></th>
          <td style="background-color: white; padding: 4px;"><input type="text" id="base_directory" name="base_directory" value="$base_directory" /></td>
        </tr>
        <tr>
          <th style="border: 1px solid white; background: #C7E7FF; padding: 4px;"><label for="base_url">Gitweb URL</label></th>
          <td style="background-color: white; padding: 4px;"><input type="text" id="base_url" name="base_url" value="$base_url" /></td>
        </tr>
        <tr>
          <td colspan="2" style="border: 1px solid white; background-color: #E3F3FF; text-align: right; padding: 4px;"><input type="submit" value="Update" class="button primary" /></td>
        </tr>
      </table>
    </form>
HTML;
} /* }}} function gitweb_show_plugin_page */

function gitweb_register_plugin_page () /* {{{ */
{
  yourls_register_plugin_page ('gitweb_page', 'Gitweb',
    'gitweb_show_plugin_page');
} /* }}} function gitweb_register_plugin_page */

yourls_add_action ('plugins_loaded', 'gitweb_register_plugin_page');
yourls_add_action ('redirect_keyword_not_found', 'gitweb_redirect_keyword_not_found');

/* vim: set sw=2 sts=2 et fdm=marker : */
?>
